# -*- coding: utf-8 -*-
"""
ldap0.extop.vc - Verify Credentials extended operations

LDAP Verify Credentials operation
"""

# Imports from pyasn1
from pyasn1.type import namedtype, univ, tag
from pyasn1.codec.der import encoder, decoder
from pyasn1_modules.rfc2251 import LDAPDN, AuthenticationChoice

from . import ExtendedRequest, ExtendedResponse


class VCRequest(ExtendedRequest):
    """
    LDAP Verify Credentials request
    """
    __slots__ = (
        'cookie',
        'name',
        'cred',
    )
    requestName: str = '1.3.6.1.4.1.4203.666.6.5'

    class VCRequestValue(univ.Sequence):
        """
        VCRequest ::= SEQUENCE {
                     cookie [0] OCTET STRING OPTIONAL,
                     name    LDAPDN,
                     authentication  AuthenticationChoice,
                 controls [2] Controls OPTIONAL
        }

        where LDAPDN, AuthenticationChoice, and Controls are as defined in RFC 4511.
        """
        tagSet = univ.Sequence.tagSet.tagImplicitly(
            tag.Tag(tag.tagClassApplication, tag.tagFormatConstructed, 0)
        )
        componentType = namedtype.NamedTypes(
            namedtype.NamedType('cookie', univ.OctetString()),
            namedtype.NamedType('name', LDAPDN()),
            namedtype.NamedType('authentication', AuthenticationChoice())
        )

    def __init__(self, name: str, cred: bytes, cookie=None):
        ExtendedRequest.__init__(self)
        self.name = name
        self.cred = cred
        self.cookie = cookie

    def encode(self) -> bytes:
        req_value = self.VCRequestValue()
        req_value.setComponentByName(
            'cookie',
            univ.OctetString(self.cookie).subtype(
                implicitTag=tag.Tag(tag.tagClassContext, tag.tagFormatSimple, 0)
            )
        )
        req_value.setComponentByName(
            'name',
            LDAPDN(self.name.encode(self.encoding)).subtype(
                implicitTag=tag.Tag(tag.tagClassContext, tag.tagFormatSimple, 1)
            )
        )
        authc = AuthenticationChoice()
        authc.setComponentByName(
            'simple',
            univ.OctetString(self.cookie).subtype(
                implicitTag=tag.Tag(tag.tagClassContext, tag.tagFormatSimple, 0)
            )
        )
        req_value.setComponentByName('authentication', authc)
        return encoder.encode(req_value)


class VCResponse(ExtendedResponse):
    """
    LDAP Verify Credentials response
    """
    __slots__ = (
        'cookie',
    )
    responseName: str = '1.3.6.1.4.1.4203.666.6.5'

    class VCResponseValue(univ.Sequence):
        """
        VCResponse ::= SEQUENCE {
                     resultCode ResultCode,
                     diagnosticMessage LDAPString,
                     cookie [0] OCTET STRING OPTIONAL,
                     serverSaslCreds [1] OCTET STRING OPTIONAL,
                 controls [2] Controls OPTIONAL
        }

        where ResultCode is the result code enumeration from RFC 4511,
        and LDAPString and Controls are as defined in RFC 4511.
        """
        tagSet = univ.Sequence.tagSet.tagImplicitly(
            tag.Tag(tag.tagClassApplication, tag.tagFormatConstructed, 0)
        )
        componentType = namedtype.NamedTypes(
            namedtype.NamedType('cookie', univ.OctetString()),
            namedtype.NamedType('serverSaslCreds', univ.OctetString()),
        )

    def decode(self, value: bytes):
        refresh_response, _ = decoder.decode(
            value,
            asn1Spec=self.VCResponseValue(),
        )
        self.cookie = refresh_response.getComponentByName('cookie')
        self.cookie = refresh_response.getComponentByName('serverSaslCreds')
        return self.cookie
