module Msf::Exploit::Remote::Pkcs12

  class Storage
    include Msf::Auxiliary::Report

    # @!attribute [r] framework
    #   @return [Msf::Framework] the Metasploit framework instance
    attr_reader :framework

    # @!attribute [r] framework_module
    #   @return [Msf::Module] the Metasploit framework module that is associated with the authentication instance
    attr_reader :framework_module

    def initialize(framework: nil, framework_module: nil)
      @framework = framework || framework_module&.framework
      @framework_module = framework_module
    end

    # Get stored pkcs12 matching the options query.
    #
    # @param [Hash] options The options for matching pkcs12's.
    # @option options [Integer, Array<Integer>] :id The identifier of the pkcs12 (optional)
    # @option options [String] :realm The realm of the pkcs12 (optional)
    # @option options [String] :username The username of the pkcs12 (optional)
    # @return [Array<StoredPkcs12>]
    def pkcs12(options = {}, &block)
      stored_pkcs12_array = filter_pkcs12(options).map do |pkcs12_entry|
        StoredPkcs12.new(pkcs12_entry)
      end

      stored_pkcs12_array.each do |stored_pkcs12|
        block.call(stored_pkcs12) if block_given?
      end

      stored_pkcs12_array
    end

    # Return the raw stored pkcs12.
    #
    # @param [Hash] options See the options hash description in {#pkcs12}.
    # @return [Array<Metasploit::Credential::Core>]
    def filter_pkcs12(options)
      return [] unless active_db?

      filter = {}
      filter[:id] = options[:id] if options[:id].present?

      creds = framework.db.creds(
        workspace: options.fetch(:workspace) { workspace },
        type: 'Metasploit::Credential::Pkcs12',
        **filter
      ).select do |cred|
        # this is needed since if a filter is provided (e.g. `id:`) framework.db.creds will ignore the type:
        next false unless cred.private.type == 'Metasploit::Credential::Pkcs12'

        if options[:username].present?
          next false if options[:username].casecmp(cred.public.username) != 0
        end

        if options[:realm].present? && cred.realm
          next false if options[:realm].casecmp(cred.realm.value) != 0
        end

        if options[:status].present?
          # If status is not set on the credential, considere it is `active`
          status = cred.private.status || 'active'
          next false if status != options[:status]
        end

        cert = cred.private.openssl_pkcs12.certificate
        unless Time.now.between?(cert.not_before, cert.not_after)
          ilog("[filter_pkcs12] Found a matching certificate but it has expired")
          next false
        end

        if options[:tls_auth]
          eku = cert.extensions.select { |c| c.oid == 'extendedKeyUsage' }.first
          unless eku&.value.include?('TLS Web Client Authentication')
            ilog("[filter_pkcs12] Found a matching certificate but it doesn't have the 'TLS Web Client Authentication' EKU")
            next false
          end
        end

        true
      end
    end

    def delete(options = {})
      if options.keys == [:ids]
        # skip calling #filter_pkcs12 which issues a query when the IDs are specified
        ids = options[:ids]
      else
        ids = filter_pkcs12(options).map(&:id)
      end

      framework.db.delete_credentials(ids: ids).map do |stored_pkcs12|
        StoredPkcs12.new(stored_pkcs12)
      end
    end

    # @return [String] The name of the workspace in which to operate.
    def workspace
      if @framework_module
        return @framework_module.workspace
      elsif @framework&.db&.active
        return @framework.db.workspace&.name
      end
    end

    # Mark Pkcs12(s) as inactive
    #
    # @param [Array<Integer>] ids The list of pkcs12 IDs.
    # @return [Array<StoredPkcs12>]
    def deactivate(ids:)
      set_status(ids: ids, status: 'inactive')
    end

    # Mark Pkcs12(s) as active
    #
    # @param [Array<Integer>] ids The list of pkcs12 IDs.
    # @return [Array<StoredPkcs12>]
    def activate(ids:)
      set_status(ids: ids, status: 'active')
    end

    private

    # @param [Array<Integer>] ids List of pkcs12 IDs to update
    # @param [String] status The status to set for the pkcs12
    # @return [Array<StoredPkcs12>]
    def set_status(ids:, status:)
      updated_pkcs12 = []
      ids.each do |id|
        pkcs12 = filter_pkcs12({ id: id })
        if pkcs12.blank?
          print_warning("Pkcs12 with id: #{id} was not found in the database")
          next
        end
        private = pkcs12.first.private
        private.metadata.merge!({ 'status' => status } )
        updated_pkcs12 << framework.db.update_credential({ id: id, private: { id: private.id, metadata: private.metadata }})
        # I know this looks weird but the local db returns a single loot object, remote db returns an array of them
        #updated_certs << Array.wrap(framework.db.update_loot({ id: id, info: updated_pkcs12_status })).first
      end
      updated_pkcs12.map do |stored_pkcs12|
        StoredPkcs12.new(stored_pkcs12)
      end
    end

  end
end
