// Aqsis
// Copyright (C) 1997 - 2001, Paul C. Gregory
//
// Contact: pgregory@aqsis.org
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


/** \file
		\brief Implements the CqCellNoise class fro producing cellular noise.
		\author Paul C. Gregory (pgregory@aqsis.org)
*/
// This is based on code authored orignally by Thomas Burge published on
//   the Affine webpages.
// Major changes were made to the hashing functions by Jonathan Merritt
//   (j.merritt@pgrad.unimelb.edu.au) on 10th Oct 2005.  These changes allow
//   the hash for all elements of a return value to depend upon all elements
//   of the input value(s).  Previously (and in the code by T Burge), there
//   were numerous instances where (for example), the hash for the
//   x-component of the output would depend only upon the x-component of the
//   input, leading to heavily coherent noise.

#include	<aqsis/aqsis.h>

#include <aqsis/math/cellnoise.h>

namespace Aqsis {

//---------------------------------------------------------------------
// Utility functions

namespace {

// Given a float, return an index into the permutation table
inline TqUint permTableIndex(TqFloat x)
{
	return static_cast<TqUint>(static_cast<TqInt>(x)) & 0x7ff;
}

} // unnamed namespace

//---------------------------------------------------------------------
/** 1D float cell noise.
 */

TqFloat CqCellNoise::FCellNoise1( TqFloat u )
{
	if ( u < 0.0 )
		u -= 1;
	TqInt i = m_PermuteTable[permTableIndex(u)];

	return ( m_RandomTable[ i ] );
}


//---------------------------------------------------------------------
/** 2D float cell noise.
 */

TqFloat CqCellNoise::FCellNoise2( TqFloat u, TqFloat v )
{
	if ( u < 0.0 )
		u -= 1;
	if ( v < 0.0 )
		v -= 1;

	TqInt i = m_PermuteTable[permTableIndex(u)];
	i = m_PermuteTable[i + permTableIndex(v)];

	return ( m_RandomTable[ i ] );
}


//---------------------------------------------------------------------
/** 3D float cell noise.
 */

TqFloat CqCellNoise::FCellNoise3( const CqVector3D& P )
{
	TqFloat x = P.x();
	TqFloat y = P.x();
	TqFloat z = P.z();

	if ( x < 0.0 )
		x -= 1;
	if ( y < 0.0 )
		y -= 1;
	if ( z < 0.0 )
		z -= 1;

	TqInt i = m_PermuteTable[permTableIndex(x)];
	i = m_PermuteTable[i + permTableIndex(y)];
	i = m_PermuteTable[i + permTableIndex(z)];

	return ( m_RandomTable[ i ] );
}


//---------------------------------------------------------------------
/** 4D float cell noise.
 */

TqFloat CqCellNoise::FCellNoise4( const CqVector3D& P, TqFloat v )
{
	TqFloat x = P.x();
	TqFloat y = P.y();
	TqFloat z = P.z();

	if ( x < 0.0 )
		x -= 1;
	if ( y < 0.0 )
		y -= 1;
	if ( z < 0.0 )
		z -= 1;
	if ( v < 0.0 )
		v -= 1;

	TqInt i = m_PermuteTable[permTableIndex(x)];
	i = m_PermuteTable[i + permTableIndex(y)];
	i = m_PermuteTable[i + permTableIndex(z)];
	i = m_PermuteTable[i + permTableIndex(v)];

	return ( m_RandomTable[ i ] );
}


//---------------------------------------------------------------------
/** 1D point cell noise.
 */

CqVector3D CqCellNoise::PCellNoise1( TqFloat u )
{
	CqVector3D result;

	if ( u < 0.0 )
		u -= 1;

	TqInt i = m_PermuteTable[permTableIndex(u)];

	result.x( m_RandomTable[ i ] );
	i = m_PermuteTable[ i ];
	result.y( m_RandomTable[ i ] );
	i = m_PermuteTable[ i ];
	result.z( m_RandomTable[ i ] );

	return ( result );
}


//---------------------------------------------------------------------
/** 2D float cell noise.
 */

CqVector3D CqCellNoise::PCellNoise2( TqFloat u, TqFloat v )
{
	CqVector3D result;

	if ( u < 0.0 )
		u -= 1;
	if ( v < 0.0 )
		v -= 1;

	TqInt i = m_PermuteTable[permTableIndex(u)];
	i = m_PermuteTable[i + permTableIndex(v)];

	result.x( m_RandomTable[ i ] );
	i = m_PermuteTable[ i ];
	result.y( m_RandomTable[ i ] );
	i = m_PermuteTable[ i ];
	result.z( m_RandomTable[ i ] );

	return ( result );
}


//---------------------------------------------------------------------
/** 3D float cell noise.
 */

CqVector3D CqCellNoise::PCellNoise3( const CqVector3D& P )
{
	CqVector3D result;

	TqFloat x = P.x();
	TqFloat y = P.y();
	TqFloat z = P.z();

	if ( x < 0.0 )
		x -= 1;
	if ( y < 0.0 )
		y -= 1;
	if ( z < 0.0 )
		z -= 1;

	TqInt i = m_PermuteTable[permTableIndex(x)];
	i = m_PermuteTable[i + permTableIndex(y)];
	i = m_PermuteTable[i + permTableIndex(z)];

	result.x( m_RandomTable[ i ] );
	i = m_PermuteTable[ i ];
	result.y( m_RandomTable[ i ] );
	i = m_PermuteTable[ i ];
	result.z( m_RandomTable[ i ] );

	return ( result );
}


//---------------------------------------------------------------------
/** 4D float cell noise.
 */

CqVector3D CqCellNoise::PCellNoise4( const CqVector3D& P, TqFloat v )
{
	CqVector3D result;

	TqFloat x = P.x();
	TqFloat y = P.y();
	TqFloat z = P.z();

	if ( x < 0.0 )
		x -= 1;
	if ( y < 0.0 )
		y -= 1;
	if ( z < 0.0 )
		z -= 1;
	if ( v < 0.0 )
		v -= 1;

	TqInt i = m_PermuteTable[permTableIndex(x)];
	i = m_PermuteTable[i + permTableIndex(y)];
	i = m_PermuteTable[i + permTableIndex(z)];
	i = m_PermuteTable[i + permTableIndex(v)];

	result.x( m_RandomTable[ i ] );
	i = m_PermuteTable[ i ];
	result.y( m_RandomTable[ i ] );
	i = m_PermuteTable[ i ];
	result.z( m_RandomTable[ i ] );

	return ( result );
}


//---------------------------------------------------------------------
/** Random permutation lookup table.
 */

/* Refer to Affine Example code permute.c to create the following
 *  table.  An array is create of integers 0 upto 2047.  
 *
 * The following code was used to create the table below with the
 * default seed value.  The table is duplicated twice as a convenient
 * way of running a permutation twice with P[P[i]+i] where i is 0..2047.
 * 
 * It's my belief that odd looking tables should have the code that 
 * created them placed nearby, so here is the code used and if there
 * are any problems send email to me -- tburge 6-17-01.
 *
 * getrandom()'s magic numbers are from Numerical Recipes in C, Second 
 * Edition, page 284. 
 *
 **********************************************************************
 * #include <stdio.h>
 * 
 * static unsigned int idum=1;
 * 
 * void setseed(unsigned int seed)
 * {
 *    idum = seed;
 * }
 * 
 * int getrandom( int max )
 * {
 *    idum = 1664525*idum + 1013904223;
 * 
 *    max++;
 * 
 *    return idum % max;
 * }
 * 
 * int main(int argc, char **argv) 
 * {
 *    unsigned int  P[2048];
 *    int           i,j,k,n;
 * 
 *    if (argc==2)
 *       setseed(atoi(argv[1]));
 * 
 *    for (i=0;i<2048;i++)
 *       P[i] = i;
 * 
 *    for (n=0;n<11;n++)
 *    {
 * 		for (i=0;i<2048;i++)
 * 		{
 * 			j = getrandom(2047);
 * 			k = P[j];
 * 			P[j] = P[i];
 * 			P[i] = k;
 * 		}
 * 	}
 * 
 *    printf("   ");   
 *    for (i=0;i<2048;i++)
 *    {
 *       printf("%4u, ",P[i]);
 *       if (i%10==9)
 * 	 printf("\n   ");
 *    }
 *    printf("\n");
 * 
 *    return 0;
 * }
 **********************************************************************
 *
 * The above code create the m_PermuteTable[] table below:
 *
 */
TqInt CqCellNoise::m_PermuteTable[ 2*2048 ] =
    {
        672, 625, 458, 751, 272, 757, 218, 1247, 636, 557,
        402, 1403, 976, 635, 898, 255, 192, 985, 178, 91,
        1748, 1181, 298, 1481, 440, 523, 1574, 815, 248, 1291,
        246, 441, 948, 1171, 582, 189, 1184, 1273, 490, 23,
        344, 409, 1202, 411, 728, 215, 286, 55, 460, 927,
        1290, 675, 712, 1161, 414, 365, 104, 641, 106, 79,
        348, 325, 626, 501, 1052, 1327, 738, 1013, 1020, 715,
        538, 1069, 1732, 97, 962, 421, 376, 165, 830, 925,
        596, 193, 242, 901, 1672, 221, 454, 1675, 1200, 969,
        354, 329, 132, 77, 786, 979, 292, 789, 1810, 765,
        1504, 503, 726, 87, 1032, 1781, 226, 475, 24, 279,
        790, 99, 708, 9, 1058, 139, 520, 1381, 998, 429,
        1192, 707, 694, 341, 1408, 107, 678, 459, 780, 717,
        306, 1777, 912, 51, 1378, 1121, 1196, 1021, 846, 1361,
        96, 1325, 686, 663, 808, 1995, 886, 453, 736, 389,
        938, 229, 1852, 651, 58, 305, 812, 689, 142, 1555,
        1208, 1299, 362, 1341, 1804, 1449, 1814, 1203, 588, 309,
        986, 157, 1368, 343, 342, 101, 552, 595, 534, 1787,
        1444, 1677, 38, 491, 804, 41, 318, 131, 1120, 465,
        418, 629, 864, 431, 862, 1331, 1068, 239, 258, 1987,
        324, 225, 1966, 863, 1016, 1377, 958, 133, 1544, 413,
        1318, 5, 232, 1267, 1258, 779, 544, 31, 778, 75,
        1236, 547, 6, 169, 1772, 1155, 518, 831, 880, 631,
        2030, 1783, 1508, 1509, 814, 1459, 604, 1101, 1962, 397,
        1160, 293, 1230, 267, 484, 893, 394, 127, 1104, 105,
        1322, 1131, 1000, 17, 1586, 881, 1560, 1721, 670, 1007,
        1816, 1185, 1398, 513, 1384, 813, 262, 1659, 180, 891,
        26, 197, 1044, 1241, 1342, 347, 528, 1437, 514, 1737,
        724, 555, 838, 1071, 648, 1547, 270, 697, 1692, 1427,
        346, 59, 176, 1529, 1502, 1315, 204, 665, 110, 1467,
        1476, 471, 290, 311, 568, 1183, 326, 931, 0, 1417,
        598, 621, 860, 897, 230, 21, 296, 581, 1382, 1463,
        992, 245, 666, 1269, 868, 241, 1582, 1091, 256, 353,
        918, 677, 936, 767, 878, 1711, 1512, 449, 126, 1157,
        1396, 477, 446, 177, 68, 1225, 906, 585, 1992, 527,
        18, 249, 444, 1045, 2, 1983, 1520, 61, 954, 867,
        100, 179, 1282, 731, 1612, 535, 610, 137, 244, 265,
        442, 395, 1832, 163, 378, 171, 740, 39, 254, 597,
        640, 363, 1618, 1877, 816, 973, 806, 111, 896, 307,
        90, 1665, 1352, 1277, 78, 1617, 1600, 43, 1482, 1775,
        660, 1485, 118, 3, 464, 645, 130, 35, 968, 907,
        162, 175, 120, 945, 710, 1811, 356, 791, 1522, 187,
        12, 1705, 850, 1287, 1980, 565, 562, 1805, 480, 599,
        1082, 357, 1572, 851, 1662, 33, 468, 615, 1354, 747,
        488, 297, 1370, 387, 20, 983, 822, 1591, 924, 687,
        102, 89, 680, 495, 10, 1413, 360, 481, 1362, 1119,
        1260, 1633, 374, 1759, 676, 669, 558, 67, 1220, 1523,
        1326, 1035, 1972, 287, 526, 331, 820, 803, 510, 425,
        1084, 1411, 1858, 1087, 4, 2009, 462, 2039, 652, 1765,
        114, 217, 268, 1357, 466, 653, 656, 549, 398, 1837,
        1492, 251, 1374, 69, 1380, 361, 1434, 15, 240, 273,
        1498, 1981, 1540, 1977, 970, 1263, 784, 1441, 50, 769,
        1148, 1029, 914, 1915, 436, 1147, 1410, 1903, 1284, 1497,
        690, 603, 328, 27, 810, 7, 252, 811, 74, 1359,
        760, 49, 1666, 953, 1460, 1683, 1094, 315, 952, 1785,
        62, 1571, 856, 921, 422, 1723, 1240, 29, 798, 567,
        224, 233, 1802, 1187, 1968, 13, 618, 877, 492, 1651,
        66, 277, 48, 837, 1138, 1719, 1564, 2003, 150, 1525,
        80, 497, 1050, 1347, 1812, 609, 1194, 933, 888, 1023,
        1638, 1967, 1108, 705, 754, 195, 396, 733, 966, 433,
        136, 1681, 278, 841, 644, 783, 1298, 505, 1984, 1301,
        390, 1729, 1696, 317, 186, 1123, 752, 435, 202, 987,
        536, 93, 1302, 393, 1216, 71, 1570, 941, 796, 419,
        266, 209, 1356, 109, 1206, 853, 1920, 619, 1190, 399,
        1292, 1229, 782, 367, 1424, 563, 1890, 1921, 64, 1533,
        1526, 207, 184, 81, 1198, 2031, 1320, 1741, 994, 259,
        388, 2019, 542, 291, 1272, 1163, 570, 1619, 1324, 1201,
        654, 57, 1720, 1047, 874, 443, 196, 45, 506, 1543,
        1100, 821, 1658, 199, 28, 1735, 206, 613, 172, 121,
        250, 289, 216, 871, 550, 1003, 88, 553, 794, 643,
        1632, 1239, 930, 1847, 1144, 943, 1238, 345, 448, 445,
        770, 1669, 504, 737, 1178, 37, 1528, 95, 322, 2015,
        1684, 1841, 1830, 323, 108, 1779, 1770, 1231, 1056, 543,
        1938, 587, 364, 1059, 1610, 681, 384, 1667, 426, 1343,
        212, 1715, 2010, 569, 1616, 2021, 370, 473, 1116, 1613,
        1030, 909, 612, 805, 1742, 231, 996, 507, 1578, 1689,
        52, 617, 1958, 271, 1860, 529, 294, 887, 476, 213,
        1182, 181, 1252, 1697, 1910, 1025, 1896, 1285, 774, 161,
        692, 2025, 1262, 2033, 392, 151, 194, 859, 1040, 283,
        54, 401, 1900, 1067, 1350, 1615, 832, 1685, 70, 1209,
        472, 1939, 858, 571, 688, 183, 406, 1827, 716, 1887,
        622, 1979, 1988, 285, 802, 125, 1080, 489, 658, 201,
        36, 269, 1110, 1863, 1372, 1907, 742, 533, 1248, 1093,
        138, 1975, 1416, 1813, 430, 1695, 1316, 753, 470, 1603,
        768, 865, 122, 1221, 1448, 1279, 1390, 561, 2040, 961,
        638, 451, 1360, 989, 1266, 2001, 580, 1937, 94, 135,
        1664, 1039, 530, 761, 956, 1557, 1646, 1985, 340, 573,
        1466, 1379, 92, 691, 1794, 1243, 584, 219, 1122, 649,
        756, 141, 1730, 1197, 428, 1901, 890, 793, 280, 1545,
        766, 19, 164, 875, 1166, 655, 1328, 203, 330, 623,
        1944, 819, 602, 1627, 2032, 1789, 274, 149, 1768, 337,
        1994, 1137, 1172, 1997, 630, 515, 1680, 1189, 182, 749,
        1480, 1419, 674, 1875, 16, 47, 566, 313, 1644, 1303,
        2034, 699, 524, 301, 382, 1799, 1912, 1077, 1074, 455,
        432, 1991, 762, 869, 512, 377, 1834, 545, 72, 1127,
        2002, 1369, 1952, 103, 1882, 899, 532, 1495, 154, 889,
        368, 1199, 614, 2043, 1392, 701, 522, 1925, 872, 223,
        198, 639, 1332, 351, 2014, 993, 112, 1803, 1070, 579,
        2012, 2035, 1838, 1487, 1488, 799, 1038, 113, 40, 73,
        1022, 937, 1596, 1563, 1470, 1599, 516, 1971, 146, 825,
        1164, 1567, 1358, 729, 1224, 1869, 978, 1165, 372, 1061,
        910, 487, 2004, 763, 1886, 1945, 408, 873, 174, 147,
        84, 785, 494, 1143, 1376, 469, 1990, 115, 1296, 1953,
        1246, 1281, 1660, 1541, 1842, 159, 1752, 963, 438, 971,
        1796, 407, 1674, 1115, 840, 65, 758, 657, 336, 1323,
        30, 1871, 316, 1941, 2046, 119, 168, 185, 234, 827,
        1464, 439, 574, 485, 628, 1889, 2042, 1597, 412, 541,
        34, 381, 1700, 745, 478, 457, 732, 525, 1130, 521,
        128, 437, 578, 1309, 560, 1349, 1042, 2041, 496, 335,
        1594, 1951, 592, 1009, 302, 1859, 1712, 2013, 1706, 1477,
        1400, 1535, 1650, 817, 200, 1217, 1702, 995, 416, 1461,
        1478, 427, 940, 1387, 1614, 205, 1156, 1295, 1474, 1017,
        1580, 211, 902, 667, 1676, 829, 1002, 1635, 1264, 947,
        714, 1, 1048, 1097, 590, 905, 1728, 1833, 214, 1453,
        1308, 295, 1338, 1049, 1868, 1801, 1242, 275, 420, 839,
        158, 911, 1668, 1935, 1294, 879, 116, 1075, 650, 1883,
        576, 123, 2038, 405, 696, 593, 946, 1393, 1412, 391,
        1506, 771, 124, 1445, 1054, 1005, 1784, 1451, 1046, 661,
        1836, 303, 82, 247, 508, 1559, 1386, 955, 1916, 1345,
        1018, 1993, 352, 1333, 1430, 711, 540, 1673, 718, 1125,
        684, 633, 338, 801, 1312, 1383, 1062, 1625, 600, 173,
        1306, 1111, 332, 1751, 1442, 1145, 1656, 117, 1750, 1149,
        960, 957, 926, 447, 664, 479, 1690, 895, 60, 607,
        834, 1249, 1072, 781, 238, 1141, 620, 1589, 950, 1743,
        452, 1055, 314, 369, 876, 1771, 818, 1193, 1064, 1819,
        586, 1855, 1012, 2037, 1090, 1081, 1532, 1823, 358, 1867,
        1628, 263, 1542, 1421, 1124, 1317, 190, 743, 260, 1019,
        134, 627, 1484, 1129, 1714, 403, 928, 1041, 1686, 1399,
        988, 725, 1694, 371, 1764, 2045, 474, 1537, 1908, 1797,
        1286, 415, 44, 1219, 1774, 1227, 140, 1099, 502, 1371,
        1552, 321, 410, 913, 2024, 1579, 1862, 721, 1344, 463,
        1974, 375, 984, 1931, 642, 1687, 1964, 695, 1790, 741,
        1228, 1083, 166, 1853, 1568, 155, 210, 637, 380, 1001,
        1170, 713, 1652, 11, 1634, 777, 1884, 693, 1254, 1565,
        1760, 1605, 350, 1235, 1928, 591, 486, 417, 1828, 1265,
        982, 261, 1280, 919, 634, 1733, 1960, 1791, 882, 1073,
        836, 1473, 1150, 1251, 1872, 1717, 1778, 683, 1092, 1643,
        606, 461, 1004, 1551, 42, 1691, 276, 467, 1114, 923,
        456, 1085, 334, 1891, 1336, 1637, 310, 1415, 1096, 1353,
        22, 1961, 208, 1795, 1854, 1709, 320, 551, 1402, 1305,
        220, 227, 386, 531, 1800, 1095, 1678, 1167, 1840, 1297,
        842, 1135, 1892, 823, 1930, 1245, 1880, 379, 1086, 83,
        1268, 849, 934, 1649, 8, 647, 1034, 1027, 828, 1701,
        1142, 1261, 2028, 1707, 1186, 917, 1232, 559, 1078, 1373,
        744, 1815, 942, 2011, 32, 1601, 894, 1443, 288, 243,
        1762, 967, 944, 1929, 1274, 1319, 1024, 1959, 1414, 1057,
        76, 1639, 1914, 1881, 908, 499, 366, 1367, 156, 2007,
        1366, 1401, 2036, 373, 1158, 1405, 1904, 1213, 222, 703,
        236, 735, 1026, 1151, 1844, 1919, 1510, 1505, 776, 1037,
        2006, 1397, 2020, 1845, 1562, 1999, 2000, 1311, 1550, 1493,
        1212, 2027, 1534, 795, 400, 1693, 1982, 577, 1028, 1583,
        1734, 1337, 160, 1313, 1870, 333, 1736, 519, 974, 1065,
        884, 1573, 1422, 999, 624, 1275, 1210, 883, 920, 167,
        1970, 659, 792, 143, 1006, 1655, 1888, 981, 1106, 153,
        1932, 951, 1758, 1793, 2008, 319, 1518, 671, 1956, 1475,
        46, 1483, 608, 1355, 1894, 129, 1060, 63, 1270, 1169,
        308, 1835, 1234, 977, 144, 719, 682, 1713, 1948, 1933,
        746, 1943, 1976, 253, 450, 997, 1140, 1339, 866, 759,
        1468, 1271, 546, 1447, 1152, 1257, 990, 1389, 1244, 1767,
        1642, 1033, 1428, 949, 498, 1821, 148, 1177, 1554, 1491,
        1008, 847, 1622, 673, 852, 1521, 1866, 517, 1500, 1175,
        1798, 1989, 1300, 2047, 282, 1329, 1756, 191, 434, 1507,
        1688, 1973, 1010, 939, 1452, 1899, 698, 1259, 56, 1807,
        1986, 1947, 424, 723, 1782, 1179, 904, 861, 1514, 1375,
        1776, 1893, 1226, 1671, 1588, 1609, 1102, 1923, 300, 709,
        2026, 1965, 1820, 807, 1850, 1561, 1864, 483, 1754, 787,
        932, 1351, 1630, 1423, 2016, 1553, 1806, 53, 1940, 1079,
        1162, 1501, 1088, 1471, 706, 339, 1876, 1105, 1458, 1905,
        1924, 903, 2018, 1283, 1188, 1957, 1566, 1517, 1604, 1963,
        1558, 1173, 1704, 1489, 594, 1629, 1364, 601, 1946, 349,
        1592, 1857, 1846, 1699, 1548, 25, 1942, 1223, 1640, 355,
        1214, 1575, 700, 1385, 1918, 679, 1824, 1895, 1746, 1587,
        284, 755, 1818, 1623, 844, 281, 1954, 1657, 964, 1107,
        1310, 1661, 228, 1469, 1438, 959, 1176, 991, 1334, 1407,
        572, 1153, 1346, 1761, 1584, 1293, 750, 1653, 1132, 1391,
        1462, 1873, 312, 2029, 826, 1749, 1388, 237, 1330, 1051,
        1576, 1949, 1098, 833, 1524, 1839, 1602, 1593, 2044, 1569,
        870, 589, 304, 775, 1426, 1321, 1636, 1829, 702, 1255,
        772, 1531, 646, 1139, 1996, 235, 1902, 915, 1440, 85,
        1250, 1911, 1076, 1237, 86, 1851, 1472, 1207, 482, 1499,
        1168, 575, 170, 1769, 556, 1731, 730, 1739, 1648, 1611,
        1014, 385, 188, 1455, 922, 1425, 500, 1581, 1490, 1233,
        1856, 975, 1922, 1969, 1496, 327, 1154, 1753, 564, 509,
        1450, 1253, 1740, 1595, 1418, 1015, 1036, 1527, 722, 1703,
        892, 1513, 1682, 1645, 1936, 2023, 1118, 1289, 1432, 1205,
        1766, 727, 972, 1433, 1822, 1747, 1556, 1103, 854, 929,
        1792, 1621, 1494, 773, 1716, 1431, 1278, 511, 1204, 257,
        1394, 1585, 1348, 1211, 1950, 1763, 764, 1519, 1710, 1195,
        1788, 1133, 98, 1515, 1516, 145, 554, 797, 788, 1053,
        1626, 1435, 1128, 1117, 1218, 1631, 1848, 1439, 1698, 1927,
        1608, 1865, 1066, 1861, 720, 965, 1606, 359, 900, 1063,
        14, 1817, 1180, 739, 1826, 1043, 916, 1607, 1530, 1679,
        404, 1809, 1174, 1365, 1708, 1335, 1126, 1757, 548, 1727,
        1598, 1897, 1780, 835, 1446, 843, 1112, 1159, 1546, 1539,
        1340, 1503, 1654, 1773, 616, 685, 1726, 1429, 1744, 1745,
        1590, 1885, 1256, 857, 1454, 605, 824, 539, 1406, 1955,
        800, 1109, 734, 1479, 1456, 611, 1786, 1831, 1536, 1641,
        1926, 935, 264, 809, 662, 1843, 1420, 1011, 1906, 1879,
        668, 537, 1878, 1913, 1808, 1363, 1670, 1917, 1436, 1725,
        1738, 1215, 748, 1755, 1538, 1663, 848, 1409, 2022, 2017,
        1288, 1549, 1314, 1909, 704, 1647, 1718, 299, 1404, 423,
        1998, 2005, 1724, 493, 1222, 1307, 152, 855, 1898, 1089,
        632, 1457, 1874, 1849, 980, 1825, 1978, 845, 1620, 1031,
        1486, 1577, 1276, 383, 1934, 1511, 1136, 1191, 1722, 1395,
        1624, 885, 1134, 583, 1304, 1113, 1146, 1465,

        672, 625, 458, 751, 272, 757, 218, 1247, 636, 557,
        402, 1403, 976, 635, 898, 255, 192, 985, 178, 91,
        1748, 1181, 298, 1481, 440, 523, 1574, 815, 248, 1291,
        246, 441, 948, 1171, 582, 189, 1184, 1273, 490, 23,
        344, 409, 1202, 411, 728, 215, 286, 55, 460, 927,
        1290, 675, 712, 1161, 414, 365, 104, 641, 106, 79,
        348, 325, 626, 501, 1052, 1327, 738, 1013, 1020, 715,
        538, 1069, 1732, 97, 962, 421, 376, 165, 830, 925,
        596, 193, 242, 901, 1672, 221, 454, 1675, 1200, 969,
        354, 329, 132, 77, 786, 979, 292, 789, 1810, 765,
        1504, 503, 726, 87, 1032, 1781, 226, 475, 24, 279,
        790, 99, 708, 9, 1058, 139, 520, 1381, 998, 429,
        1192, 707, 694, 341, 1408, 107, 678, 459, 780, 717,
        306, 1777, 912, 51, 1378, 1121, 1196, 1021, 846, 1361,
        96, 1325, 686, 663, 808, 1995, 886, 453, 736, 389,
        938, 229, 1852, 651, 58, 305, 812, 689, 142, 1555,
        1208, 1299, 362, 1341, 1804, 1449, 1814, 1203, 588, 309,
        986, 157, 1368, 343, 342, 101, 552, 595, 534, 1787,
        1444, 1677, 38, 491, 804, 41, 318, 131, 1120, 465,
        418, 629, 864, 431, 862, 1331, 1068, 239, 258, 1987,
        324, 225, 1966, 863, 1016, 1377, 958, 133, 1544, 413,
        1318, 5, 232, 1267, 1258, 779, 544, 31, 778, 75,
        1236, 547, 6, 169, 1772, 1155, 518, 831, 880, 631,
        2030, 1783, 1508, 1509, 814, 1459, 604, 1101, 1962, 397,
        1160, 293, 1230, 267, 484, 893, 394, 127, 1104, 105,
        1322, 1131, 1000, 17, 1586, 881, 1560, 1721, 670, 1007,
        1816, 1185, 1398, 513, 1384, 813, 262, 1659, 180, 891,
        26, 197, 1044, 1241, 1342, 347, 528, 1437, 514, 1737,
        724, 555, 838, 1071, 648, 1547, 270, 697, 1692, 1427,
        346, 59, 176, 1529, 1502, 1315, 204, 665, 110, 1467,
        1476, 471, 290, 311, 568, 1183, 326, 931, 0, 1417,
        598, 621, 860, 897, 230, 21, 296, 581, 1382, 1463,
        992, 245, 666, 1269, 868, 241, 1582, 1091, 256, 353,
        918, 677, 936, 767, 878, 1711, 1512, 449, 126, 1157,
        1396, 477, 446, 177, 68, 1225, 906, 585, 1992, 527,
        18, 249, 444, 1045, 2, 1983, 1520, 61, 954, 867,
        100, 179, 1282, 731, 1612, 535, 610, 137, 244, 265,
        442, 395, 1832, 163, 378, 171, 740, 39, 254, 597,
        640, 363, 1618, 1877, 816, 973, 806, 111, 896, 307,
        90, 1665, 1352, 1277, 78, 1617, 1600, 43, 1482, 1775,
        660, 1485, 118, 3, 464, 645, 130, 35, 968, 907,
        162, 175, 120, 945, 710, 1811, 356, 791, 1522, 187,
        12, 1705, 850, 1287, 1980, 565, 562, 1805, 480, 599,
        1082, 357, 1572, 851, 1662, 33, 468, 615, 1354, 747,
        488, 297, 1370, 387, 20, 983, 822, 1591, 924, 687,
        102, 89, 680, 495, 10, 1413, 360, 481, 1362, 1119,
        1260, 1633, 374, 1759, 676, 669, 558, 67, 1220, 1523,
        1326, 1035, 1972, 287, 526, 331, 820, 803, 510, 425,
        1084, 1411, 1858, 1087, 4, 2009, 462, 2039, 652, 1765,
        114, 217, 268, 1357, 466, 653, 656, 549, 398, 1837,
        1492, 251, 1374, 69, 1380, 361, 1434, 15, 240, 273,
        1498, 1981, 1540, 1977, 970, 1263, 784, 1441, 50, 769,
        1148, 1029, 914, 1915, 436, 1147, 1410, 1903, 1284, 1497,
        690, 603, 328, 27, 810, 7, 252, 811, 74, 1359,
        760, 49, 1666, 953, 1460, 1683, 1094, 315, 952, 1785,
        62, 1571, 856, 921, 422, 1723, 1240, 29, 798, 567,
        224, 233, 1802, 1187, 1968, 13, 618, 877, 492, 1651,
        66, 277, 48, 837, 1138, 1719, 1564, 2003, 150, 1525,
        80, 497, 1050, 1347, 1812, 609, 1194, 933, 888, 1023,
        1638, 1967, 1108, 705, 754, 195, 396, 733, 966, 433,
        136, 1681, 278, 841, 644, 783, 1298, 505, 1984, 1301,
        390, 1729, 1696, 317, 186, 1123, 752, 435, 202, 987,
        536, 93, 1302, 393, 1216, 71, 1570, 941, 796, 419,
        266, 209, 1356, 109, 1206, 853, 1920, 619, 1190, 399,
        1292, 1229, 782, 367, 1424, 563, 1890, 1921, 64, 1533,
        1526, 207, 184, 81, 1198, 2031, 1320, 1741, 994, 259,
        388, 2019, 542, 291, 1272, 1163, 570, 1619, 1324, 1201,
        654, 57, 1720, 1047, 874, 443, 196, 45, 506, 1543,
        1100, 821, 1658, 199, 28, 1735, 206, 613, 172, 121,
        250, 289, 216, 871, 550, 1003, 88, 553, 794, 643,
        1632, 1239, 930, 1847, 1144, 943, 1238, 345, 448, 445,
        770, 1669, 504, 737, 1178, 37, 1528, 95, 322, 2015,
        1684, 1841, 1830, 323, 108, 1779, 1770, 1231, 1056, 543,
        1938, 587, 364, 1059, 1610, 681, 384, 1667, 426, 1343,
        212, 1715, 2010, 569, 1616, 2021, 370, 473, 1116, 1613,
        1030, 909, 612, 805, 1742, 231, 996, 507, 1578, 1689,
        52, 617, 1958, 271, 1860, 529, 294, 887, 476, 213,
        1182, 181, 1252, 1697, 1910, 1025, 1896, 1285, 774, 161,
        692, 2025, 1262, 2033, 392, 151, 194, 859, 1040, 283,
        54, 401, 1900, 1067, 1350, 1615, 832, 1685, 70, 1209,
        472, 1939, 858, 571, 688, 183, 406, 1827, 716, 1887,
        622, 1979, 1988, 285, 802, 125, 1080, 489, 658, 201,
        36, 269, 1110, 1863, 1372, 1907, 742, 533, 1248, 1093,
        138, 1975, 1416, 1813, 430, 1695, 1316, 753, 470, 1603,
        768, 865, 122, 1221, 1448, 1279, 1390, 561, 2040, 961,
        638, 451, 1360, 989, 1266, 2001, 580, 1937, 94, 135,
        1664, 1039, 530, 761, 956, 1557, 1646, 1985, 340, 573,
        1466, 1379, 92, 691, 1794, 1243, 584, 219, 1122, 649,
        756, 141, 1730, 1197, 428, 1901, 890, 793, 280, 1545,
        766, 19, 164, 875, 1166, 655, 1328, 203, 330, 623,
        1944, 819, 602, 1627, 2032, 1789, 274, 149, 1768, 337,
        1994, 1137, 1172, 1997, 630, 515, 1680, 1189, 182, 749,
        1480, 1419, 674, 1875, 16, 47, 566, 313, 1644, 1303,
        2034, 699, 524, 301, 382, 1799, 1912, 1077, 1074, 455,
        432, 1991, 762, 869, 512, 377, 1834, 545, 72, 1127,
        2002, 1369, 1952, 103, 1882, 899, 532, 1495, 154, 889,
        368, 1199, 614, 2043, 1392, 701, 522, 1925, 872, 223,
        198, 639, 1332, 351, 2014, 993, 112, 1803, 1070, 579,
        2012, 2035, 1838, 1487, 1488, 799, 1038, 113, 40, 73,
        1022, 937, 1596, 1563, 1470, 1599, 516, 1971, 146, 825,
        1164, 1567, 1358, 729, 1224, 1869, 978, 1165, 372, 1061,
        910, 487, 2004, 763, 1886, 1945, 408, 873, 174, 147,
        84, 785, 494, 1143, 1376, 469, 1990, 115, 1296, 1953,
        1246, 1281, 1660, 1541, 1842, 159, 1752, 963, 438, 971,
        1796, 407, 1674, 1115, 840, 65, 758, 657, 336, 1323,
        30, 1871, 316, 1941, 2046, 119, 168, 185, 234, 827,
        1464, 439, 574, 485, 628, 1889, 2042, 1597, 412, 541,
        34, 381, 1700, 745, 478, 457, 732, 525, 1130, 521,
        128, 437, 578, 1309, 560, 1349, 1042, 2041, 496, 335,
        1594, 1951, 592, 1009, 302, 1859, 1712, 2013, 1706, 1477,
        1400, 1535, 1650, 817, 200, 1217, 1702, 995, 416, 1461,
        1478, 427, 940, 1387, 1614, 205, 1156, 1295, 1474, 1017,
        1580, 211, 902, 667, 1676, 829, 1002, 1635, 1264, 947,
        714, 1, 1048, 1097, 590, 905, 1728, 1833, 214, 1453,
        1308, 295, 1338, 1049, 1868, 1801, 1242, 275, 420, 839,
        158, 911, 1668, 1935, 1294, 879, 116, 1075, 650, 1883,
        576, 123, 2038, 405, 696, 593, 946, 1393, 1412, 391,
        1506, 771, 124, 1445, 1054, 1005, 1784, 1451, 1046, 661,
        1836, 303, 82, 247, 508, 1559, 1386, 955, 1916, 1345,
        1018, 1993, 352, 1333, 1430, 711, 540, 1673, 718, 1125,
        684, 633, 338, 801, 1312, 1383, 1062, 1625, 600, 173,
        1306, 1111, 332, 1751, 1442, 1145, 1656, 117, 1750, 1149,
        960, 957, 926, 447, 664, 479, 1690, 895, 60, 607,
        834, 1249, 1072, 781, 238, 1141, 620, 1589, 950, 1743,
        452, 1055, 314, 369, 876, 1771, 818, 1193, 1064, 1819,
        586, 1855, 1012, 2037, 1090, 1081, 1532, 1823, 358, 1867,
        1628, 263, 1542, 1421, 1124, 1317, 190, 743, 260, 1019,
        134, 627, 1484, 1129, 1714, 403, 928, 1041, 1686, 1399,
        988, 725, 1694, 371, 1764, 2045, 474, 1537, 1908, 1797,
        1286, 415, 44, 1219, 1774, 1227, 140, 1099, 502, 1371,
        1552, 321, 410, 913, 2024, 1579, 1862, 721, 1344, 463,
        1974, 375, 984, 1931, 642, 1687, 1964, 695, 1790, 741,
        1228, 1083, 166, 1853, 1568, 155, 210, 637, 380, 1001,
        1170, 713, 1652, 11, 1634, 777, 1884, 693, 1254, 1565,
        1760, 1605, 350, 1235, 1928, 591, 486, 417, 1828, 1265,
        982, 261, 1280, 919, 634, 1733, 1960, 1791, 882, 1073,
        836, 1473, 1150, 1251, 1872, 1717, 1778, 683, 1092, 1643,
        606, 461, 1004, 1551, 42, 1691, 276, 467, 1114, 923,
        456, 1085, 334, 1891, 1336, 1637, 310, 1415, 1096, 1353,
        22, 1961, 208, 1795, 1854, 1709, 320, 551, 1402, 1305,
        220, 227, 386, 531, 1800, 1095, 1678, 1167, 1840, 1297,
        842, 1135, 1892, 823, 1930, 1245, 1880, 379, 1086, 83,
        1268, 849, 934, 1649, 8, 647, 1034, 1027, 828, 1701,
        1142, 1261, 2028, 1707, 1186, 917, 1232, 559, 1078, 1373,
        744, 1815, 942, 2011, 32, 1601, 894, 1443, 288, 243,
        1762, 967, 944, 1929, 1274, 1319, 1024, 1959, 1414, 1057,
        76, 1639, 1914, 1881, 908, 499, 366, 1367, 156, 2007,
        1366, 1401, 2036, 373, 1158, 1405, 1904, 1213, 222, 703,
        236, 735, 1026, 1151, 1844, 1919, 1510, 1505, 776, 1037,
        2006, 1397, 2020, 1845, 1562, 1999, 2000, 1311, 1550, 1493,
        1212, 2027, 1534, 795, 400, 1693, 1982, 577, 1028, 1583,
        1734, 1337, 160, 1313, 1870, 333, 1736, 519, 974, 1065,
        884, 1573, 1422, 999, 624, 1275, 1210, 883, 920, 167,
        1970, 659, 792, 143, 1006, 1655, 1888, 981, 1106, 153,
        1932, 951, 1758, 1793, 2008, 319, 1518, 671, 1956, 1475,
        46, 1483, 608, 1355, 1894, 129, 1060, 63, 1270, 1169,
        308, 1835, 1234, 977, 144, 719, 682, 1713, 1948, 1933,
        746, 1943, 1976, 253, 450, 997, 1140, 1339, 866, 759,
        1468, 1271, 546, 1447, 1152, 1257, 990, 1389, 1244, 1767,
        1642, 1033, 1428, 949, 498, 1821, 148, 1177, 1554, 1491,
        1008, 847, 1622, 673, 852, 1521, 1866, 517, 1500, 1175,
        1798, 1989, 1300, 2047, 282, 1329, 1756, 191, 434, 1507,
        1688, 1973, 1010, 939, 1452, 1899, 698, 1259, 56, 1807,
        1986, 1947, 424, 723, 1782, 1179, 904, 861, 1514, 1375,
        1776, 1893, 1226, 1671, 1588, 1609, 1102, 1923, 300, 709,
        2026, 1965, 1820, 807, 1850, 1561, 1864, 483, 1754, 787,
        932, 1351, 1630, 1423, 2016, 1553, 1806, 53, 1940, 1079,
        1162, 1501, 1088, 1471, 706, 339, 1876, 1105, 1458, 1905,
        1924, 903, 2018, 1283, 1188, 1957, 1566, 1517, 1604, 1963,
        1558, 1173, 1704, 1489, 594, 1629, 1364, 601, 1946, 349,
        1592, 1857, 1846, 1699, 1548, 25, 1942, 1223, 1640, 355,
        1214, 1575, 700, 1385, 1918, 679, 1824, 1895, 1746, 1587,
        284, 755, 1818, 1623, 844, 281, 1954, 1657, 964, 1107,
        1310, 1661, 228, 1469, 1438, 959, 1176, 991, 1334, 1407,
        572, 1153, 1346, 1761, 1584, 1293, 750, 1653, 1132, 1391,
        1462, 1873, 312, 2029, 826, 1749, 1388, 237, 1330, 1051,
        1576, 1949, 1098, 833, 1524, 1839, 1602, 1593, 2044, 1569,
        870, 589, 304, 775, 1426, 1321, 1636, 1829, 702, 1255,
        772, 1531, 646, 1139, 1996, 235, 1902, 915, 1440, 85,
        1250, 1911, 1076, 1237, 86, 1851, 1472, 1207, 482, 1499,
        1168, 575, 170, 1769, 556, 1731, 730, 1739, 1648, 1611,
        1014, 385, 188, 1455, 922, 1425, 500, 1581, 1490, 1233,
        1856, 975, 1922, 1969, 1496, 327, 1154, 1753, 564, 509,
        1450, 1253, 1740, 1595, 1418, 1015, 1036, 1527, 722, 1703,
        892, 1513, 1682, 1645, 1936, 2023, 1118, 1289, 1432, 1205,
        1766, 727, 972, 1433, 1822, 1747, 1556, 1103, 854, 929,
        1792, 1621, 1494, 773, 1716, 1431, 1278, 511, 1204, 257,
        1394, 1585, 1348, 1211, 1950, 1763, 764, 1519, 1710, 1195,
        1788, 1133, 98, 1515, 1516, 145, 554, 797, 788, 1053,
        1626, 1435, 1128, 1117, 1218, 1631, 1848, 1439, 1698, 1927,
        1608, 1865, 1066, 1861, 720, 965, 1606, 359, 900, 1063,
        14, 1817, 1180, 739, 1826, 1043, 916, 1607, 1530, 1679,
        404, 1809, 1174, 1365, 1708, 1335, 1126, 1757, 548, 1727,
        1598, 1897, 1780, 835, 1446, 843, 1112, 1159, 1546, 1539,
        1340, 1503, 1654, 1773, 616, 685, 1726, 1429, 1744, 1745,
        1590, 1885, 1256, 857, 1454, 605, 824, 539, 1406, 1955,
        800, 1109, 734, 1479, 1456, 611, 1786, 1831, 1536, 1641,
        1926, 935, 264, 809, 662, 1843, 1420, 1011, 1906, 1879,
        668, 537, 1878, 1913, 1808, 1363, 1670, 1917, 1436, 1725,
        1738, 1215, 748, 1755, 1538, 1663, 848, 1409, 2022, 2017,
        1288, 1549, 1314, 1909, 704, 1647, 1718, 299, 1404, 423,
        1998, 2005, 1724, 493, 1222, 1307, 152, 855, 1898, 1089,
        632, 1457, 1874, 1849, 980, 1825, 1978, 845, 1620, 1031,
        1486, 1577, 1276, 383, 1934, 1511, 1136, 1191, 1722, 1395,
        1624, 885, 1134, 583, 1304, 1113, 1146, 1465
    };


//---------------------------------------------------------------------
/** Random float value lookup table.
 */

TqFloat CqCellNoise::m_RandomTable[ 2048 ] =
    {
        0.4729110505, 0.7385413474, 0.008484064601, 0.4097665274, 0.1010872573,
        0.7390367086, 0.549525925, 0.1123771411, 0.03298647143, 0.2784920796,
        0.500902283, 0.8447555364, 0.1813803948, 0.6738673183, 0.4701518388,
        0.9616919239, 0.7217741888, 0.6537510059, 0.3652054435, 0.06291822577,
        0.4318840131, 0.2090684543, 0.1410902003, 0.6377613959, 0.2596301902,
        0.4144928395, 0.1657741917, 0.7586248652, 0.5258959942, 0.5018094587,
        0.8614018308, 0.354548112, 0.6683020648, 0.9664831138, 0.7771698376,
        0.09609435732, 0.9322464466, 0.9886786188, 0.750093841, 0.422816385,
        0.9154602531, 0.9499844327, 0.309915605, 0.7445504875, 0.3723289557,
        0.3270894173, 0.9844008805, 0.3477885355, 0.1842292976, 0.7436627201,
        0.6612778818, 0.03837817395, 0.9021318331, 0.4616867821, 0.6630663881,
        0.05180114182, 0.7677190918, 0.09349347046, 0.6910474701, 0.2622797177,
        0.6192693673, 0.8157533365, 0.2946238415, 0.2219612901, 0.5885561313,
        0.8666679985, 0.02232083585, 0.06143105263, 0.9950130731, 0.6076156492,
        0.9106780076, 0.7828028244, 0.343414763, 0.9304996892, 0.4672813592,
        0.9766379953, 0.8313229419, 0.2920504841, 0.8041861923, 0.4939425965,
        0.2725484353, 0.1563302479, 0.07798058446, 0.1044924953, 0.8428786919,
        0.1266910504, 0.8927394906, 0.672736716, 0.5543209482, 0.5484359502,
        0.8221194698, 0.8825288848, 0.8641110174, 0.8634332526, 0.7068440365,
        0.04193680035, 0.3247377034, 0.4979338921, 0.8838296374, 0.9993801289,
        0.6812328398, 0.5648419405, 0.003202253021, 0.7023462248, 0.3219769541,
        0.1616882398, 0.5894546239, 0.4299340756, 0.4892705567, 0.5455571595,
        0.002994102426, 0.2304763724, 0.155921055, 0.9662164799, 0.9583311295,
        0.5954897054, 0.9740952924, 0.4387003998, 0.2551583359, 0.901202884,
        0.2026877124, 0.2365956693, 0.8785702689, 0.6489133821, 0.01945414767,
        0.3872808474, 0.1246737437, 0.03542190837, 0.1241612341, 0.9503686852,
        0.9078388205, 0.8847951018, 0.03900144994, 0.3606016333, 0.9058404053,
        0.4727063519, 0.01252377965, 0.6164504751, 0.699140477, 0.2746047438,
        0.9333300032, 0.595666084, 0.5605609389, 0.1689621289, 0.1597866099,
        0.2790343068, 0.05159709509, 0.12684832, 0.6720023379, 0.1635839711,
        0.08162175212, 0.9190844116, 0.4523213375, 0.6464980184, 0.5862299455,
        0.8722242252, 0.5005819611, 0.6609777617, 0.4809742915, 0.2046138798,
        0.3903911766, 0.3453102685, 0.04674096312, 0.973779487, 0.7726646066,
        0.02638886822, 0.4030162925, 0.1663929713, 0.7326821927, 0.2989367857,
        0.2252939967, 0.9620660157, 0.4069094844, 0.4816838983, 0.3628730299,
        0.7022221317, 0.7658345532, 0.2317466191, 0.5132960593, 0.5952968546,
        0.9690705314, 0.5984609048, 0.6096785674, 0.6895160642, 0.1989737768,
        0.2979183882, 0.07717857789, 0.1445024298, 0.3790276609, 0.4894556361,
        0.6148570077, 0.3329599802, 0.6832419839, 0.8353465325, 0.6591786435,
        0.8036867999, 0.2428180128, 0.1249477859, 0.1854037428, 0.6371658486,
        0.9563291464, 0.2445475622, 0.003120261244, 0.2249839068, 0.8096672036,
        0.7741672429, 0.2021667035, 0.004328646231, 0.6120043378, 0.9925663075,
        0.9051956004, 0.1788138249, 0.5540205762, 0.5717003415, 0.9831276098,
        0.9568826077, 0.4946469832, 0.7418821692, 0.8897462441, 0.3390390114,
        0.3825744502, 0.2088961112, 0.2716832208, 0.9851574753, 0.7187781092,
        0.604315368, 0.5100444397, 0.1931205564, 0.4663073719, 0.7502725269,
        0.8499265844, 0.5200347495, 0.3135915939, 0.5199331953, 0.2740743486,
        0.07727443287, 0.6975113638, 0.5749227586, 0.7769282879, 0.03060393827,
        0.4924854171, 0.7609562627, 0.1953132963, 0.3366277465, 0.7718231753,
        0.442921422, 0.2520177076, 0.2468188074, 0.54759291, 0.560664752,
        0.9685130222, 0.6103749857, 0.8952555321, 0.6867308277, 0.1030642586,
        0.5071536307, 0.3693140056, 0.8673046655, 0.7704809299, 0.241934272,
        0.1162283868, 0.5276072319, 0.8997586453, 0.2312594396, 0.5908464883,
        0.2229940756, 0.685861337, 0.8141036094, 0.2826001681, 0.517007086,
        0.6918941112, 0.5176202734, 0.357717244, 0.7676293752, 0.7579167252,
        0.809217135, 0.6236990318, 0.1029932867, 0.3726476887, 0.8662094292,
        0.7223135103, 0.3677915274, 0.6643089904, 0.3943013684, 0.9573541097,
        0.3216414261, 0.6668614494, 0.02614571201, 0.6634141263, 0.8706523157,
        0.01788069215, 0.3312314623, 0.5219267607, 0.6134460415, 0.7443237538,
        0.9683496845, 0.7307207827, 0.4828805937, 0.2924363045, 0.0118874521,
        0.4333418198, 0.7647268656, 0.4580505015, 0.9831999172, 0.3143892717,
        0.2745485357, 0.3735234654, 0.618310506, 0.7671188489, 0.9740350763,
        0.2075663814, 0.9031892386, 0.5395607371, 0.8080378859, 0.7342142267,
        0.407859175, 0.2654033862, 0.04355954705, 0.4271861212, 0.4505809848,
        0.7858929243, 0.8919462091, 0.2358863046, 0.123365643, 0.6689981967,
        0.6955330963, 0.6992957378, 0.7100652498, 0.8320914302, 0.4599931403,
        0.5540043293, 0.5282919765, 0.6743945219, 0.01363873808, 0.492633787,
        0.7264151354, 0.6254409533, 0.5749266348, 0.2289010948, 0.06688637519,
        0.5158033296, 0.009415931534, 0.5255726138, 0.2271430013, 0.6763365436,
        0.5574292182, 0.3415588094, 0.6493394212, 0.1721865647, 0.3136668312,
        0.7543996023, 0.4701567148, 0.07781485282, 0.2400242365, 0.8143903455,
        0.5619403296, 0.199288398, 0.992862537, 0.9865984255, 0.2164159012,
        0.150085384, 0.3459821059, 0.3370204894, 0.5022331984, 0.1866664775,
        0.4905493441, 0.1191302957, 0.8275881936, 0.710061742, 0.9932265631,
        0.9170671916, 0.7391732582, 0.8397231773, 0.6937549929, 0.001715213992,
        0.4887054828, 0.965892883, 0.3232409148, 0.05592281651, 0.3982831226,
        0.6867368482, 0.1243949193, 0.9251132207, 0.5558201675, 0.03649436496,
        0.2549742884, 0.5495353928, 0.8718090379, 0.4108930677, 0.2556785042,
        0.7344056452, 0.02866140427, 0.09607959725, 0.3637478813, 0.914189863,
        0.3538771332, 0.8073158897, 0.9535153243, 0.5672763558, 0.1482913573,
        0.1436910909, 0.8851612112, 0.4372893339, 0.5006496706, 0.3650990948,
        0.04292140109, 0.2172930101, 0.1197614488, 0.3977317084, 0.8439983665,
        0.8531363262, 0.2154812948, 0.47433744, 0.9994254517, 0.1221883046,
        0.9597686087, 0.3154959138, 0.3081342555, 0.6437140247, 0.5590752834,
        0.2581579089, 0.7654751386, 0.9771412248, 0.4693115246, 0.237701023,
        0.7674408569, 0.9644675283, 0.7846397967, 0.02975096181, 0.6918433192,
        0.9730582954, 0.3313468923, 0.658018453, 0.6376934699, 0.1950850999,
        0.4980441281, 0.3744017854, 0.6040342967, 0.6598548302, 0.333445665,
        0.1176049728, 0.8894769638, 0.1152794166, 0.9429749534, 0.8565215431,
        0.9936900693, 0.434736182, 0.7154493039, 0.7246339526, 0.8020920283,
        0.7055688193, 0.4110925817, 0.3516786247, 0.3349965806, 0.6554789925,
        0.642126028, 0.2988266144, 0.8424614235, 0.5730585633, 0.7771872045,
        0.00362989679, 0.5260899696, 0.3787124427, 0.8008270641, 0.1409832817,
        0.6690362296, 0.002222067676, 0.1593351462, 0.8063519523, 0.4554628204,
        0.7231995026, 0.1241428968, 0.4275146294, 0.2607020973, 0.6306821657,
        0.7040551142, 0.8110723607, 0.6933704126, 0.8581181811, 0.6375595187,
        0.2299729344, 0.1706854976, 0.7499923548, 0.4964644485, 0.9583009183,
        0.308236971, 0.6163427765, 0.4321146798, 0.1595833246, 0.9054599716,
        0.2314180462, 0.5954353721, 0.5348608978, 0.8080619159, 0.7327251257,
        0.7620541384, 0.6367750298, 0.4286523811, 0.07671781536, 0.1937512406,
        0.2559202984, 0.2068261127, 0.7073758552, 0.2674879101, 0.7857126314,
        0.7899491577, 0.5939246519, 0.9032627926, 0.9720230959, 0.2158981222,
        0.2939456301, 0.3220465188, 0.9538273122, 0.8790025539, 0.1981272092,
        0.1650443994, 0.001120492816, 0.5604404504, 0.6129064048, 0.5055372459,
        0.8563481066, 0.304336892, 0.8372510551, 0.7845503152, 0.08560216799,
        0.4208157989, 0.8898326727, 0.2016003807, 0.3458331507, 0.3972548149,
        0.04283992108, 0.591770927, 0.974327527, 0.9989933143, 0.8186827162,
        0.3203654704, 0.8067009319, 0.3407435357, 0.605836953, 0.2263890044,
        0.6296569519, 0.2099548387, 0.550079857, 0.1461075698, 0.1747615393,
        0.4233810878, 0.8773659607, 0.0479359501, 0.0594830513, 0.4981011595,
        0.3045937056, 0.3099034918, 0.5817479137, 0.4182425751, 0.6945126681,
        0.1709480495, 0.7742619105, 0.7786582592, 0.6110214116, 0.8872880568,
        0.6248470973, 0.08680642163, 0.931105963, 0.6251227208, 0.8690443411,
        0.5040835659, 0.1696689138, 0.6209103432, 0.2612314839, 0.8078606217,
        0.6734991474, 0.6404747651, 0.7305071317, 0.8554722029, 0.34063219,
        0.2681899373, 0.3275900688, 0.3314089901, 0.02135118749, 0.05749261612,
        0.3689842969, 0.05896323873, 0.257078507, 0.07397318305, 0.684659699,
        0.6575774117, 0.5134079633, 0.8622945906, 0.3754983284, 0.8271459374,
        0.5635276707, 0.3681894322, 0.9867563229, 0.04052009201, 0.1782835191,
        0.8467851379, 0.503791444, 0.925466008, 0.7791332761, 0.2885072944,
        0.07630624063, 0.1173258065, 0.2102413131, 0.3938110904, 0.8773568384,
        0.8636318552, 0.2859639572, 0.627990887, 0.003377353773, 0.1619241512,
        0.2699754676, 0.3873179131, 0.8214699328, 0.7120208289, 0.9422846371,
        0.8077560854, 0.6702104155, 0.4639917924, 0.410392019, 0.247505954,
        0.3202196248, 0.04308547406, 0.320837778, 0.9746267726, 0.1007502805,
        0.8328587166, 0.6273901677, 0.09098669374, 0.09853953868, 0.9977597813,
        0.5721246311, 0.2237910894, 0.3351771552, 0.2264144807, 0.03564352822,
        0.01593899587, 0.3292392753, 0.4768654364, 0.9126864867, 0.9464576845,
        0.9494378474, 0.5050144619, 0.6693813791, 0.5122653567, 0.9650356621,
        0.4575180705, 0.7383859241, 0.3025119645, 0.1997770164, 0.3103861078,
        0.9081699131, 0.9967481554, 0.6955840029, 0.4345335099, 0.3626880618,
        0.8181723435, 0.7922684234, 0.06953518605, 0.5276934868, 0.473205904,
        0.5295093283, 0.9868340264, 0.3799418071, 0.1086464939, 0.2773461808,
        0.1236520349, 0.3755358709, 0.3176494972, 0.001437645406, 0.4688545871,
        0.6536611067, 0.7257875954, 0.06933600269, 0.9820178128, 0.6719615003,
        0.1884384626, 0.004027187824, 0.8269491042, 0.9298061961, 0.1307505858,
        0.09098645486, 0.7009107484, 0.9356967239, 0.06143721053, 0.2449996658,
        0.5407966855, 0.07506485004, 0.7916484107, 0.04300020449, 0.387512391,
        0.5347110545, 0.3901712843, 0.329171516, 0.6898078839, 0.9401336322,
        0.4062082735, 0.2985652704, 0.8288138597, 0.8619984193, 0.3909836099,
        0.4654851519, 0.1445309478, 0.8479991453, 0.2494278853, 0.4228862729,
        0.2455220944, 0.1363584092, 0.4531907463, 0.7990402132, 0.8830580707,
        0.7072192812, 0.6462317151, 0.317715684, 0.1709661051, 0.8281775294,
        0.6742391349, 0.3681248166, 0.4324316415, 0.7502044709, 0.5691272398,
        0.990885077, 0.455003222, 0.7102104491, 0.5198613503, 0.6862468049,
        0.4350504945, 0.8964989344, 0.3608611939, 0.9508363772, 0.3929253458,
        0.5333612515, 0.6092323498, 0.9491649978, 0.3400761592, 0.7410619305,
        0.5820758198, 0.2261531111, 0.9793651975, 0.3274646839, 0.6251257234,
        0.8669204414, 0.2198391105, 0.1675490076, 0.9839651179, 0.01009165682,
        0.2872025683, 0.3270674078, 0.3490699441, 0.1208676584, 0.7112450185,
        0.5865932526, 0.6059310962, 0.9301124942, 0.9714705697, 0.5222092485,
        0.8215046176, 0.4457058981, 0.08218099875, 0.7990865419, 0.9982400206,
        0.9424114544, 0.8982618838, 0.8342044456, 0.6269367323, 0.3365103863,
        0.422921862, 0.4844607906, 0.5696786647, 0.8514509704, 0.8986350126,
        0.9164229026, 0.3040969097, 0.3807635754, 0.9625363643, 0.3139692033,
        0.06021055998, 0.4544913266, 0.6475205286, 0.5800714372, 0.8810956539,
        0.2153863944, 0.5102934572, 0.6889675232, 0.1386322458, 0.3111219089,
        0.6676273248, 0.844975031, 0.5355796018, 0.1087694988, 0.02206853731,
        0.1041952828, 0.1252295193, 0.137792455, 0.4583488642, 0.6153580798,
        0.3798877732, 0.1678045206, 0.2917496967, 0.1360570537, 0.8394336854,
        0.827344304, 0.7496694303, 0.9806256434, 0.3712078207, 0.1699153185,
        0.7676439728, 0.05590369646, 0.5724778068, 0.09354164638, 0.8810703768,
        0.1411446957, 0.3467322798, 0.02008922771, 0.4938835981, 0.06831361074,
        0.1850529541, 0.7406027745, 0.3054363779, 0.9590253746, 0.1838148185,
        0.83290831, 0.1768322508, 0.1743739853, 0.3299644557, 0.5577313323,
        0.2179616778, 0.1339022322, 0.085126488, 0.6395806037, 0.376491304,
        0.6599620478, 0.7997881477, 0.8386475649, 0.3100563413, 0.003715484403,
        0.9888119721, 0.7199713737, 0.8229060764, 0.2090263152, 0.9995266241,
        0.5261195693, 0.6482525035, 0.9704922615, 0.1036458374, 0.5596343614,
        0.857572082, 0.6418678435, 0.5442654868, 0.9814945441, 0.6781106195,
        0.5511278575, 0.5691887834, 0.4318430349, 0.9998412798, 0.2784214774,
        0.9818295953, 0.3792300392, 0.3531257152, 0.05316698086, 0.2409438924,
        0.6046218388, 0.6383283245, 0.9265371012, 0.6404393171, 0.7264252063,
        0.3886336363, 0.8755850038, 0.6005128794, 0.1727774441, 0.847316816,
        0.4952608878, 0.6013667774, 0.5073520485, 0.6407172992, 0.4345692126,
        0.7907523424, 0.5149299987, 0.3281655493, 0.2331165494, 0.7965921094,
        0.9529866707, 0.6101129404, 0.7143000579, 0.7759300121, 0.3755443767,
        0.475714765, 0.5913865874, 0.2314597038, 0.935540529, 0.07111636689,
        0.442732404, 0.626881002, 0.571961429, 0.5696923989, 0.7123935418,
        0.3323344593, 0.4879916813, 0.8254892058, 0.8923767796, 0.9311640584,
        0.326429233, 0.09112033201, 0.5427812422, 0.4192580325, 0.9486160371,
        0.5813651546, 0.3060806273, 0.3282178105, 0.2231343426, 0.1637206408,
        0.5717960568, 0.3036050876, 0.7305186074, 0.9571562312, 0.9478095537,
        0.1695170202, 0.7902330384, 0.120321847, 0.1945681432, 0.01073954673,
        0.7161647547, 0.6104590236, 0.7784285797, 0.3037288324, 0.7068825774,
        0.1942651677, 0.7003926979, 0.6276708399, 0.2769114431, 0.4919609404,
        0.75648339, 0.9868690246, 0.6352724433, 0.3358107652, 0.8860112065,
        0.2757101548, 0.9175384883, 0.2243464603, 0.7639815835, 0.9174582572,
        0.6776555963, 0.1535439971, 0.2938926937, 0.208126687, 0.5457798075,
        0.6062033824, 0.1571739232, 0.3966783234, 0.4583797231, 0.9807816665,
        0.07551497314, 0.03280388983, 0.3668546695, 0.240929076, 0.9423355861,
        0.6135283993, 0.8309184872, 0.06697827158, 0.4796458976, 0.05989572452,
        0.4029989447, 0.2906367308, 0.5765474131, 0.05498129362, 0.2099027783,
        0.894132256, 0.965624609, 0.7744829101, 0.6379963551, 0.355061607,
        0.3934771596, 0.04113969812, 0.5281519629, 0.6181922588, 0.9416427882,
        0.4342291732, 0.786709724, 0.4754672847, 0.6541607445, 0.3853924372,
        0.8187051341, 0.635463364, 0.1280259052, 0.7920374987, 0.6896122266,
        0.2635519146, 0.2227835013, 0.1795707182, 0.4218523614, 0.2740133512,
        0.545514469, 0.9437063788, 0.3323186394, 0.1553544677, 0.8674614271,
        0.7041043811, 0.817165941, 0.6100789453, 0.1285452778, 0.3005860639,
        0.4901165906, 0.7901609852, 0.1860224335, 0.463178149, 0.08053164557,
        0.4094720664, 0.9635326685, 0.687234812, 0.9974978101, 0.5144108287,
        0.1567875957, 0.3448878773, 0.9661799744, 0.1939860801, 0.1521497676,
        0.5640751389, 0.6426893789, 0.0106029301, 0.3143524034, 0.9063968048,
        0.6137174033, 0.4328020471, 0.2996097403, 0.3750500684, 0.6871849801,
        0.05120204156, 0.5503680101, 0.7840918028, 0.8801576346, 0.8588949735,
        0.6279470408, 0.02027938189, 0.0102761481, 0.3775449856, 0.5393224834,
        0.2287396253, 0.2969528474, 0.9104400608, 0.7142986627, 0.4537158688,
        0.8785835188, 0.7037469051, 0.7893220549, 0.7656431594, 0.6520249471,
        0.2972189556, 0.8541853884, 0.4058015761, 0.3405880798, 0.8456075131,
        0.3178723706, 0.9797976748, 0.1967968158, 0.692026, 0.04979303945,
        0.2311240588, 0.2461624201, 0.9745173431, 0.9526597401, 0.4260470741,
        0.4782235622, 0.5470744814, 0.6232537264, 0.8810726297, 0.8911019806,
        0.9963639989, 0.2573323874, 0.6642401428, 0.7958237715, 0.03540288238,
        0.4549270989, 0.001494609285, 0.2866563033, 0.05543840444, 0.08228170034,
        0.419386799, 0.2837107554, 0.117281131, 0.8466815958, 0.1553515759,
        0.05406028219, 0.1633410542, 0.740423481, 0.8668143437, 0.6176238023,
        0.7316168505, 0.01017569471, 0.1703756158, 0.9440654125, 0.9528882629,
        0.8079791954, 0.04232371086, 0.346959427, 0.1123616095, 0.1801391626,
        0.611725573, 0.9814710412, 0.5570184155, 0.5502572237, 0.377345717,
        0.851802621, 0.2299171719, 0.3526543053, 0.3796432992, 0.2346556168,
        0.6126982174, 0.97243411, 0.3590973117, 0.9249581024, 0.3575417469,
        0.6484372178, 0.432109904, 0.2101079393, 0.3898289441, 0.4953358127,
        0.3157378272, 0.9789815657, 0.7627562392, 0.3011190342, 0.632502005,
        0.8719584104, 0.04515269166, 0.256218954, 0.3265252286, 0.878210932,
        0.5237504984, 0.2705271775, 0.7222479773, 0.2865102831, 0.001158626284,
        0.03455207031, 0.2569723637, 0.3957975991, 0.4707895699, 0.4809666099,
        0.418484204, 0.8917709123, 0.4499892346, 0.8027968621, 0.9190032133,
        0.2957376912, 0.252619944, 0.6845010826, 0.636711326, 0.3919830043,
        0.9823465734, 0.9021894438, 0.3561091409, 0.0399033092, 0.5278758067,
        0.949234386, 0.8384504919, 0.2771558668, 0.3413985199, 0.8435215941,
        0.2535051429, 0.1200737804, 0.2814852889, 0.7725915993, 0.5038996083,
        0.9675966594, 0.3016850217, 0.7329220055, 0.4732860983, 0.01496690884,
        0.2660767171, 0.8196003409, 0.7295895894, 0.583441643, 0.6730243466,
        0.3226375347, 0.7145830193, 0.772270225, 0.568384863, 0.2861720184,
        0.9510406074, 0.3391874302, 0.4294193326, 0.6868144525, 0.2987193004,
        0.2156722359, 0.3006571936, 0.8873830782, 0.7904183422, 0.5632222351,
        0.9629636384, 0.5222796937, 0.07936696941, 0.7768936753, 0.4170601214,
        0.470756256, 0.02919023158, 0.3423571233, 0.4627479925, 0.07432210725,
        0.4777029366, 0.9527187534, 0.655085254, 0.2545418264, 0.705781721,
        0.7913572621, 0.4187664441, 0.687473706, 0.6425652453, 0.3870264068,
        0.6019556443, 0.6909988532, 0.338232514, 0.9474731702, 0.2508226126,
        0.9812909169, 0.7356093214, 0.577821549, 0.8860320407, 0.9546764391,
        0.2718528216, 0.2899719682, 0.06243593758, 0.6511358311, 0.3414642238,
        0.2092663199, 0.4932191367, 0.05569497589, 0.1518796128, 0.8846686278,
        0.5197479292, 0.8939574538, 0.002888894174, 0.1087113358, 0.2083718213,
        0.5780273946, 0.5211328645, 0.1534572225, 0.8554284177, 0.459072954,
        0.880928407, 0.8287269399, 0.1817014287, 0.04278041515, 0.5426718113,
        0.2688421775, 0.9975818037, 0.3239489151, 0.5400702194, 0.8541344516,
        0.6201819885, 0.8964757966, 0.8474511956, 0.1734695192, 0.8236533082,
        0.9949825844, 0.8584004217, 0.4340374172, 0.6039564754, 0.124331587,
        0.5069775176, 0.224658126, 0.5393352224, 0.4332831232, 0.06275998661,
        0.03885285929, 0.02774305968, 0.4885433735, 0.1309652855, 0.4639309775,
        0.182473158, 0.605452857, 0.8889010199, 0.4423425272, 0.6672413568,
        0.3915655324, 0.08999326499, 0.511546405, 0.2519722967, 0.6593685253,
        0.8666430102, 0.4286441468, 0.3705386776, 0.364392479, 0.8631607299,
        0.08602934144, 0.4617036409, 0.7249625316, 0.7300290787, 0.1243026704,
        0.374516278, 0.1798399175, 0.5107437116, 0.1487025712, 0.6194102629,
        0.3400041936, 0.9524822296, 0.9553015269, 0.7462556097, 0.5908260914,
        0.2718512495, 0.6732179802, 0.6306135557, 0.500880993, 0.4069615998,
        0.229093872, 0.9494445692, 0.6936530331, 0.2870816407, 0.04008780606,
        0.6275194823, 0.3383798255, 0.1511187092, 0.3416338377, 0.5357837486,
        0.9162878143, 0.4461786901, 0.05621588789, 0.2229248262, 0.4184682509,
        0.3374537104, 0.6094077826, 0.9613926853, 0.6316437954, 0.3606879604,
        0.5994289164, 0.8891234412, 0.6680757431, 0.2484716387, 0.7264655419,
        0.5282161119, 0.3958618762, 0.4616130223, 0.8880076837, 0.4618125013,
        0.9258749401, 0.4568386623, 0.8464516774, 0.4504302065, 0.8115917454,
        0.222071704, 0.3752183337, 0.768998751, 0.1180628939, 0.1105383323,
        0.2896367274, 0.04584884224, 0.01627313066, 0.5049523418, 0.2688436117,
        0.3849047548, 0.05914232042, 0.3430278325, 0.8749484569, 0.05232983781,
        0.795417049, 0.03557002963, 0.6757141147, 0.5088751479, 0.8777028499,
        0.8083942295, 0.8770031072, 0.0691812071, 0.3208870562, 0.9993228693,
        0.3712395411, 0.9692828855, 0.06720227655, 0.3415107229, 0.6081366464,
        0.1235674093, 0.5141804973, 0.7643482205, 0.193939941, 0.3524354934,
        0.1617478086, 0.7431928948, 0.6254360192, 0.3619155991, 0.03470975626,
        0.7291789507, 0.06497696601, 0.2564813173, 0.03687731829, 0.6903549605,
        0.5627703965, 0.8663218762, 0.8930835063, 0.2953996849, 0.6326851193,
        0.6703157737, 0.8353405921, 0.7711660177, 0.5878224261, 0.5959575358,
        0.6893879594, 0.9653030564, 0.5421030056, 0.477487599, 0.5178805804,
        0.6452062973, 0.4841194823, 0.4534753184, 0.476468117, 0.564545956,
        0.3295045812, 0.08510510018, 0.03901464771, 0.3286234126, 0.358064916,
        0.4764156598, 0.2482090192, 0.5898057823, 0.9419304039, 0.1777424044,
        0.1478158226, 0.6042886102, 0.9709693044, 0.153612941, 0.05281699542,
        0.6814366919, 0.8817188833, 0.5963457092, 0.8136940477, 0.556953019,
        0.6960817017, 0.8666017386, 0.7311009141, 0.2211258491, 0.9760375302,
        0.3421395323, 0.2771774298, 0.2334039011, 0.1006208882, 0.4560230742,
        0.2797534326, 0.0545496759, 0.7714142073, 0.705532962, 0.7256934894,
        0.4276209013, 0.152806621, 0.9130306649, 0.8396322476, 0.3390301024,
        0.5532683786, 0.5200259066, 0.5943504116, 0.5909322971, 0.05389088392,
        0.195697173, 0.3090413446, 0.5161908162, 0.9904423598, 0.541115345,
        0.491718295, 0.8671712489, 0.695189137, 0.1704204562, 0.5820122482,
        0.4096135129, 0.4047325682, 0.9502015761, 0.7506104028, 0.25283205,
        0.7401115671, 0.6783298175, 0.4115852723, 0.4474941459, 0.6654207725,
        0.9834944694, 0.6038644118, 0.8822263763, 0.3311943375, 0.7266922551,
        0.8980907695, 0.01029917644, 0.7088053431, 0.6859197146, 0.9851236632,
        0.4376395368, 0.422130838, 0.8053187127, 0.6024303818, 0.9034218234,
        0.682732502, 0.7900250717, 0.9546184121, 0.684486723, 0.7348029353,
        0.3279703292, 0.28435829, 0.9548383136, 0.7161275726, 0.719886621,
        0.7499470776, 0.1314692595, 0.3413122222, 0.1987646888, 0.2657624641,
        0.7377641317, 0.3135262299, 0.7200150131, 0.461857493, 0.8157308535,
        0.8710112683, 0.503582071, 0.4189459132, 0.4183102702, 0.3746057507,
        0.109261719, 0.3350281557, 0.213013845, 0.3425348699, 0.3264717008,
        0.7799079353, 0.7281394792, 0.838716263, 0.65972528, 0.693849477,
        0.2728154571, 0.6208312996, 0.6910579982, 0.7866253676, 0.0621225168,
        0.9544119965, 0.1006575576, 0.4932050323, 0.5785440453, 0.4991935566,
        0.6269468288, 0.1423338363, 0.7009943989, 0.1739825327, 0.7473699837,
        0.4942528112, 0.6326753483, 0.4062906168, 0.3611317533, 0.3038090067,
        0.1590106492, 0.6729840543, 0.2551398105, 0.06520646717, 0.2669087374,
        0.7381986827, 0.6344002779, 0.5946478331, 0.6566035491, 0.4947705958,
        0.4981163605, 0.6071165679, 0.1773234666, 0.815432433, 0.1427401039,
        0.943549809, 0.7180383881, 0.3200252932, 0.5732272766, 0.6046518451,
        0.5845349231, 0.4649762735, 0.1037453781, 0.2475982169, 0.8940697131,
        0.8612916339, 0.9290847718, 0.3018804444, 0.01885087648, 0.2273155861,
        0.9480229425, 0.3605413781, 0.6095975791, 0.8825047072, 0.6198918796,
        0.003051175736, 0.2304277294, 0.1884679794, 0.1355985845, 0.7060739035,
        0.1363237989, 0.8435224537, 0.6843487038, 0.9983747629, 0.2244267673,
        0.4370145388, 0.09732328681, 0.5161179425, 0.690408193, 0.1696569156,
        0.6495586685, 0.114805094, 0.4211704084, 0.1461310759, 0.3012794773,
        0.694104434, 0.6551248399, 0.1462054867, 0.159934768, 0.8918204708,
        0.9412984536, 0.2806193791, 0.4441872793, 0.3032543557, 0.9285183256,
        0.4379821625, 0.731105817, 0.3822048595, 0.01592433685, 0.9289384633,
        0.7678107615, 0.6798599744, 0.3960599941, 0.2338487897, 0.6288227006,
        0.5778291402, 0.5217913748, 0.26023582, 0.5003641876, 0.1715660961,
        0.5283041974, 0.01624631695, 0.872864624, 0.4604323572, 0.6414290168,
        0.1063786671, 0.4229182671, 0.5006513158, 0.1035445449, 0.9557260443,
        0.365945363, 0.6774029685, 0.6483090953, 0.1689851992, 0.5608064043,
        0.7523092674, 0.05547015416, 0.9304808732, 0.1476195133, 0.8425600966,
        0.8170020753, 0.8514558375, 3.094924614e-005, 0.9879298741, 0.4458219004,
        0.170876557, 0.7731887414, 0.4619347779, 0.4583333046, 0.7160044797,
        0.828669447, 0.483380598, 0.5620486666, 0.5289829504, 0.8176080338,
        0.4845512128, 0.07953926409, 0.56568975, 0.203280888, 0.5922876904,
        0.1399635938, 0.3731052596, 0.5043919473, 0.4782080157, 0.6694238172,
        0.1514462493, 0.5403119582, 0.2343174415, 0.7114003305, 0.1073152926,
        0.4595208592, 0.4302594652, 0.1085144603, 0.504115805, 0.8325032885,
        0.008457430638, 0.07686928334, 0.3159827087, 0.5903472169, 0.1733967541,
        0.7042185725, 0.8914787211, 0.09029570548, 0.9312977763, 0.9032138665,
        0.533306269, 0.08950178837, 0.4364292389, 0.8509796658, 0.4002850279,
        0.9082364808, 0.8003123859, 0.4462667522, 0.6378450859, 0.5637999116,
        0.5200656904, 0.8154336349, 0.1432857327, 0.1563855815, 0.182140206,
        0.3985061203, 0.8720498737, 0.2881477452, 0.5976955434, 0.6465650075,
        0.09126406908, 0.7967219199, 0.02586647216, 0.8617119086, 0.4867941979,
        0.5844485178, 0.6411701469, 0.210863194, 0.5300667472, 0.8245496037,
        0.9013039591, 0.4446220626, 0.01090572588, 0.3255087784, 0.9715840137,
        0.3524787375, 0.1426352039, 0.334904186, 0.8622742305, 0.4855887857,
        0.1456974763, 0.5639446466, 0.4350560708, 0.1783795641, 0.7160368524,
        0.7139589149, 0.9348888891, 0.4002347891, 0.2844990697, 0.286088062,
        0.2035109093, 0.4685072326, 0.4734963328, 0.955430984, 0.2308308268,
        0.1541095418, 0.6571879778, 0.2908930345, 0.2004600344, 0.2109331847,
        0.0314267613, 0.1019877824, 0.6856354745, 0.8603455205, 0.09972077422,
        0.1938418946, 0.1517831152, 0.2620149949, 0.9814609811, 0.8116531442,
        0.4219720522, 0.50231446, 0.448664384, 0.5558515429, 0.261509235,
        0.1315526203, 0.09748281166, 0.04921359895, 0.7379248189, 0.7812836026,
        0.5607273336, 0.1371797002, 0.5125981411, 0.8929446512, 0.1676582992,
        0.9026246215, 0.7202743227, 0.08907194855, 0.9572910015, 0.2763311579,
        0.5927214948, 0.2183445604, 0.4514669664, 0.5244168821, 0.4827786507,
        0.6057065059, 0.09394086711, 0.3939640275, 0.4450441217, 0.5388169535,
        0.7617330626, 0.1981024961, 0.02945467923, 0.5220785406, 0.2549977619,
        0.621795991, 0.4440052612, 0.3294938193, 0.1716568135, 0.5295812744,
        0.7428593235, 0.3875998263, 0.07304911874, 0.5565105504, 0.1960998168,
        0.5197307789, 0.3468456715, 0.763546973, 0.4973707879, 0.5828588945,
        0.673480222, 0.1386632784, 0.9656596063, 0.02826333651, 0.5023455657,
        0.2248935434, 0.3974698158, 0.9172772844, 0.4440054577, 0.6565880678,
        0.7256346112, 0.4233631291, 0.9846721515, 0.8851213823, 0.1410209397,
        0.3517657933, 0.4291995633, 0.8753033816, 0.8333742274, 0.7079941765,
        0.478842061, 0.05374358175, 0.007553731091, 0.846380726, 0.3500288781,
        0.2904185033, 0.3314085146, 0.2299691313, 0.8404041529, 0.1946921791,
        0.4715440506, 0.3329801452, 0.2483825926, 0.5070397868, 0.8732294468,
        0.7171055288, 0.5524707623, 0.867724407, 0.4406952979, 0.8128801216,
        0.7566039991, 0.743676906, 0.2740721619, 0.4373820373, 0.807748951,
        0.7948289434, 0.1192196617, 0.5795109835, 0.9920217413, 0.4611491053,
        0.6866962118, 0.4840947078, 0.2155717574, 0.05163216172, 0.4961193698,
        0.5660735671, 0.07648106478, 0.1164914495, 0.4021668127, 0.1860132241,
        0.1339294463, 0.3838043804, 0.4584679389, 0.8181954664, 0.2808750663,
        0.04186107358, 0.2756360946, 0.6425371529, 0.6265007593, 0.6484661135,
        0.5296914252, 0.09158728318, 0.7946759444, 0.4484434328, 0.7771243518,
        0.3838521936, 0.04468416423, 0.3805943201, 0.2328496734, 0.5746988771,
        0.1204662491, 0.5553585528, 0.6673012925, 0.1560864286, 0.2346291058,
        0.4844286549, 0.07891756762, 0.7363756155, 0.09347013198, 0.8435672796,
        0.2982672891, 0.8314665728, 0.3692672551, 0.04987771763, 0.180066281,
        0.2985146237, 0.5260845963, 0.4347775294, 0.5393346278, 0.4434742243,
        0.4053437598, 0.2939253035, 0.4879445257, 0.3337210347, 0.4774107467,
        0.5953505388, 0.3278087592, 0.3470021137, 0.1654484943, 0.627036653,
        0.6569598028, 0.4879642394, 0.1477850396, 0.3652071147, 0.8447695477,
        0.5035103546, 0.04504730925, 0.8445647019, 0.5325596528, 0.3282341366,
        0.3983105328, 0.3117338843, 0.3158525219, 0.8911826932, 0.3445132822,
        0.4431563313, 0.2645129049, 0.8151973826, 0.8953881357, 0.408689545,
        0.4370301245, 0.04008275876, 0.226157967, 0.06215861766, 0.04519092757,
        0.9008536418, 0.8802478593, 0.04011111474, 0.4253929621, 0.6923103132,
        0.2962626442, 0.05002508266, 0.4728488782, 0.2511918363, 0.5634800661,
        0.1292201816, 0.6949310554, 0.5871575014, 0.812109571, 0.1557219434,
        0.5399069535, 0.09384639608, 0.1445667204, 0.3923291541, 0.1573150894,
        0.3712651008, 0.5140132308, 0.3451343621, 0.2462316388, 0.1907535088,
        0.4564126376, 0.7177475109, 0.1476776628, 0.6337787719, 0.08235158399,
        0.7424726966, 0.837416959, 0.9358073701, 0.2348767351, 0.6696880511,
        0.9753945386, 0.0665053078, 0.2195973173, 0.6966787498, 0.6680657277,
        0.5775164333, 0.01330328174, 0.1171731441, 0.09986589383, 0.7490688846,
        0.357349094, 0.9728320199, 0.6900707977, 0.5666596596, 0.6420531478,
        0.9879409797, 0.9312975695, 0.5590673112, 0.9883628041, 0.06855595065,
        0.5658963108, 0.028942768, 0.4330350552, 0.1473518177, 0.2564823208,
        0.7072282792, 0.6235828139, 0.6554241809, 0.4068589481, 0.3626544853,
        0.9292368274, 0.4022942386, 0.2895810669, 0.397565613, 0.3741732165,
        0.1453667949, 0.1364091793, 0.9613227961, 0.2992566396, 0.6300913929,
        0.3479041411, 0.61253552, 0.1585490815, 0.3820765722, 0.4784125136,
        0.0613711928, 0.3568342682, 0.03240725165, 0.1526925405, 0.02319284016,
        0.5344077051, 0.4574152376, 0.5704307491, 0.7147749257, 0.2053823415,
        0.5141528561, 0.7548991768, 0.02434729645, 0.1557598598, 0.6528197112,
        0.2020032229, 0.8867172529, 0.5075175893, 0.1873925678, 0.08597012516,
        0.8947197446, 0.8549610451, 0.5057083587, 0.6779280612, 0.6782456902,
        0.3795920927, 0.0002503185533, 0.1336258752, 0.08209842304, 0.3497427367,
        0.0008872407489, 0.3065434294, 0.6739282613,
    };

} // namespace Aqsis
//---------------------------------------------------------------------
